% gravity_sim
%
% Top level function which runs the gravity simulation. Inputs are defined
% in a script or function which refers to this function.
%
% LAST UPDATED by Andy French May 2012.
%
% Syntax: gravity_sim( masses, rings, clusters, view_option, title_str )
%
% masses(n).name              = Object string e.g. 'Earth'
% masses(n).mass              = Object mass in Solar masses
% masses(n).radii             = Object radii in AU
% masses(n).x0                = Object initial x coordinate in AU
% masses(n).y0                = Object initial y coordinate in AU
% masses(n).vz0               = Object initial z coordinate in AU
% masses(n).vx0               = Object initial x velocity coordinate in AU
% masses(n).vy0               = Object initial y velocity coordinate in AU
% masses(n).vz0               = Object initial z velocity coordinate in AU
% masses(n).marker_RGB        = Fix object colour. If [] then colour by speed
%
% rings(n).xc                    = x coordinate in AU of ring centre
% rings(n).yc                    = y coordinate in AU of ring centre
% rings(n).zc                    = z coordinate in AU of ring centre
% rings(n).vxc                   = Initial ring x velocity in AU per year
% rings(n).vyc                   = Initial ring y velocity in AU per year
% rings(n).vzc                   = Initial ring z velocity in AU per year
% rings(n).num_rings             = Number of rings of massless planets
% rings(n).arc_separation_AU     = Arc separation of planet in AU
% rings(n).first_ring_radius_AU  = Radius of the first ring in AU
% rings(n).ring_radius_diff_AU   = Difference in AU between rings
% rings(n).mass_at_centre        = Mass at centre of ring in Solar masses
% masses(n).marker_RGB           = Fix object colour. If [] then colour by speed
%
% clusters(n).xc                 = x coordinate in AU of cluster centre
% clusters(n).yc                 = y coordinate in AU of cluster centre
% clusters(n).zc                 = z coordinate in AU of cluster centre
% clusters(n).vxc                = Initial cluster x velocity in AU per year
% clusters(n).vyc                = Initial cluster y velocity in AU per year
% clusters(n).vzc                = Initial cluster z velocity in AU per year
% clusters(n).shell_separation_AU       = Separation of mass cluster spherical shells /AU
% clusters(n).num_shells                = Number of spherical shells
% clusters(n).num_masses_per_square_AU  = Number of planents per square AU in each shell
% clusters(n).first_shell_radius        = Radius of first shell in AU
% clusters(n).mass_at_centre        = Mass at centre of cluster in Solar masses
% clusters(n).marker_RGB            = Fix object colour. If [] then colour by speed
%
% view_option                View option. 2 means 2D view, 3 means 3D
%                            view, [az,el] means a prescribed azimuth and elevation in degrees.
%                            Essesntially the inputs to the view command.
%
% AUmax                      +/- plot limits in AU
% vmax                       Velocity limit in AU per year for colorbar.
%                            Note masses are coloured by their speed.
% title_str                  Title string

function gravity_sim( masses, rings, clusters, view_option, AUmax, vmax, title_str )

%%% INPUTS %%%

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_sun = 2e30;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_sun/(AU^3);

%Strength of gravity
G = 1;

%Timestep / years
dt = 0.01;

%Marker size for planets
msize = 2;


%%

% Assemble arrays for all masses and massless objects
x0 = [];
y0 = [];
z0 = [];
vx0 = [];
vy0 = [];
vz0 = [];
R = [];
M = [];

% Masses
if ~isempty(masses)
    for n=1:length(masses)
        x0 = [x0,masses(n).x0];
        y0 = [y0,masses(n).y0];
        z0 = [z0,masses(n).z0];
        vx0 = [vx0,masses(n).vx0];
        vy0 = [vy0,masses(n).vy0];
        vz0 = [vz0,masses(n).vz0];
        R = [R,masses(n).radii];
        M = [M,masses(n).mass];
    end
end

%Rings of massless planets
if ~isempty(rings)
    for n=1:length(rings)
        [xx,yy,zz,vxx,vyy,vzz,num_masses] = mass_rings( rings(n).xc,rings(n).yc,rings(n).zc,....
            rings(n).mass_at_centre, rings(n).arc_separation_AU,...
            rings(n).num_rings, rings(n).first_ring_radius_AU, rings(n).ring_radius_diff_AU, P);
        x0 = [x0,xx];
        y0 = [y0,yy];
        z0 = [z0,zz];
        vx0 = [vx0,vxx + rings(n).vxc];
        vy0 = [vy0,vyy + rings(n).vyc];
        vz0 = [vz0,vzz + rings(n).vzc];
        R = [R,zeros(1,num_masses)];
        M = [M,zeros(1,num_masses)];
        rings(n).num_masses = num_masses;
    end
end

%Globular clusters of massless planets
if ~isempty(clusters)
    for n=1:length(clusters)
        [xx,yy,zz,vxx,vyy,vzz,num_masses] = globular_cluster( clusters(n).xc,clusters(n).yc,clusters(n).zc,...
            clusters(n).mass_at_centre,clusters(n).shell_separation_AU,...
            clusters(n).num_shells, clusters(n).first_shell_radius, clusters(n).num_masses_per_square_AU, P);
        x0 = [x0,xx];
        y0 = [y0,yy];
        z0 = [z0,zz];
        vx0 = [vx0,vxx + clusters(n).vxc];
        vy0 = [vy0,vyy + clusters(n).vyc];
        vz0 = [vz0,vzz + clusters(n).vzc];
        R = [R,zeros(1,num_masses)];
        M = [M,zeros(1,num_masses)];
        clusters(n).num_masses = num_masses;
    end
end

%Coefficients of restitutia (MODIFY THIS, and bounce condition - TOO COMPLEX FOR THIS SIMULATION)
C = ones(length(R),length(R));

%%

%% GRAVITY SIM ANIMATION %%

%Set up figure
fig = figure('color',[1 1 1],'name','gravity simulation animation',...
    'renderer','painters','KeyPressFcn',@keypressfunc,'units','normalized','position',[0 0 1 1]);
d = get(fig,'userdata');
d.run = 1;
d.save = 0;
d.print = 0;
d.wait = 0;
t = 0;
set(fig,'userdata',d);

%Set up axis
cmap = speed_colormap;
colormap(cmap);
colorbar
caxis([0,vmax])
hold on;
xlabel('x /AU')
ylabel('y /AU')
zlabel('z /AU')
tit = title({['Gravity sim: ',title_str],['t = ',num2str(t),' years.']});
view(view_option)
axis vis3d
axis equal
xlim([-AUmax,AUmax])
ylim([-AUmax,AUmax])
zlim([-AUmax,AUmax])
grid on;
hold on;

%Initialise object data vectors
x = x0;
y = y0;
z = z0;
vx = vx0;
vy = vy0;
vz = vz0;

%Plot initial positions. Colour code by velocity

%Masses
if ~isempty(masses)
    for n=1:length(masses)
        if isempty( masses(n).marker_RGB )
            [r,g,b] = x_to_color(sqrt(vx(n).^2 + vy(n).^2 + vz(n).^2 ),cmap,0,vmax);
        else
            r = masses(n).marker_RGB(1);
            g = masses(n).marker_RGB(2);
            b = masses(n).marker_RGB(3);
        end
        p(n) = plot3(x0(n),y0(n),z0(n),'ro','markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Rings
if ~isempty(rings)
    n0 = length(masses);
    end_index = n0;
    for n=1:length(rings)
        start_index = end_index+1;
        end_index = start_index + rings(n).num_masses - 1;
        if isempty( rings(n).marker_RGB )
            [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
        else
            r = rings(n).marker_RGB(1);
            g = rings(n).marker_RGB(2);
            b = rings(n).marker_RGB(3);
        end
        
        p(n0+n) = plot3(x0(start_index:end_index),y0(start_index:end_index),z0(start_index:end_index),'r.','markersize',msize,...
            'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Clusters
if ~isempty(clusters)
    n0 = n0 + length(rings);
    for n=1:length(clusters)
        start_index = end_index+1;
        end_index = start_index + clusters(n).num_masses - 1;
        if isempty( clusters(n).marker_RGB )
            [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
        else
            r = clusters(n).marker_RGB(1);
            g = clusters(n).marker_RGB(2);
            b = clusters(n).marker_RGB(3);
        end
        p(n0+n) = plot3(x0(start_index:end_index),y0(start_index:end_index),z0(start_index:end_index),'r.','markersize',msize,...
            'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Run simulation until user presses 'q'
while d.run == 1
    %Obtain figure user data and check for key presses
    d = get(fig,'userdata');
    if d.wait == 1
        pause(0.1);
    elseif d.wait == 0
        
        %Update positions, veocities and accelerations and times
        [ x, y, z, vx, vy, vz, ax, ay, az, collisions ] =...
            gravity( x, y, z, vx, vy, vz, R, M, C, G, dt, 2 );
        t = t + dt;
        x = x(2,:);
        y = y(2,:);
        z = z(2,:);
        vx = vx(2,:);
        vy = vy(2,:);
        vz = vz(2,:);
        
        %Masses
        if ~isempty(masses)
            for n=1:length(masses)
                if isempty( masses(n).marker_RGB )
                    [r,g,b] = x_to_color(sqrt(vx(n).^2 + vy(n).^2 + vz(n).^2 ),cmap,0,vmax);
                else
                    r = masses(n).marker_RGB(1);
                    g = masses(n).marker_RGB(2);
                    b = masses(n).marker_RGB(3);
                end
                set( p(n), 'Xdata', x(n), 'Ydata', y(n), 'Zdata', z(n),'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
                plot3(x(n),y(n),z(n),'r.','markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b],'markersize',1);
            end
        end
        
        %Rings
        if ~isempty(rings)
            n0 = length(masses);
            end_index = n0;
            for n=1:length(rings)
                start_index = end_index+1;
                end_index = start_index + rings(n).num_masses - 1;
                if isempty( rings(n).marker_RGB )
                    [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                        vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
                else
                    r = rings(n).marker_RGB(1);
                    g = rings(n).marker_RGB(2);
                    b = rings(n).marker_RGB(3);
                end
                set( p(n0+n), 'Xdata', x(start_index:end_index), 'Ydata', y(start_index:end_index), 'Zdata',...
                    z(start_index:end_index),...
                    'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
            end
        end
        
        %Clusters
        if ~isempty(clusters)
            n0 = n0 + length(rings);
            for n=1:length(clusters)
                start_index = end_index+1;
                end_index = start_index + clusters(n).num_masses - 1;
                if isempty( clusters(n).marker_RGB )
                    [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                        vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
                else
                    r = clusters(n).marker_RGB(1);
                    g = clusters(n).marker_RGB(2);
                    b = clusters(n).marker_RGB(3);
                end
                set( p(n0+n), 'Xdata', x(start_index:end_index), 'Ydata', y(start_index:end_index), 'Zdata',...
                    z(start_index:end_index),...
                    'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
            end
        end
        
        %Update title
        set(tit,'string',{['Gravity sim: ',title_str],['t = ',num2str(t),' years.']})
        
        %Flush pending graphics requests
        drawnow
        
        %Special actions resulting from key presses
        d = get(fig,'userdata');
        if d.save == 1
            %Save a .mat file of the current data set
            save( ['gravity_sim_data t=',num2str(t),'.mat'] );
            d.save = 0;
        end
        if d.print == 1;
            %Print a screenshot of the current view
            print( fig, '-dpng','-r300',['gravity sim. t=',num2str(t),...
                '.  ',strrep(datestr(now),':','-'),'.png'])
            d.print = 0;
        end
        set(fig,'userdata',d);
    end
end
close(fig);

%%

%Figure key press function callback
function keypressfunc( fig,evnt )
d = get(fig,'userdata');
if evnt.Character == 'q'
    d.run = 0;
elseif evnt.Character == 'p'
    d.print = 1;
elseif evnt.Character == 's'
    d.save = 1;
elseif evnt.Character == 'w'
    d.wait = 1;
elseif evnt.Character == 'c'
    d.wait = 0;
end
set(fig,'userdata',d);

%End of code