%gravity
% N-body gravity simulator. Uses the Verlet algorithm + Newtonian gravity
% to determine future x,y,z coordinates (and velocities and accelerations)
% of N masses.
%
% [ t, x, y, z, vx, vy, vz, ax, ay, az, bounce ] =...
%    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, C, G, k, dt, N )
%
% t               Vector of simulation times / Earth years
% x,y,z           x,y,z coordinates in AU (distance between the Earth and
%                 the Sun
% vx,vy,vz        Cartesian velocities in AU per Earth year
% ax,ay,az        Cartesian accelerations in AU per square Earth years.
% collisions      collisions(i,j) = 1 implies object i is
%                 colliding with object j
% x0,y0,z0        Initial object positions in AU
% vx0,vy0,vz0     Initial ibject velocities in AU per Earth year
% R               Row vector of object raddi / Sun radii
% M               Row vector of object masses / Sun masses
% C               Symmetric matrix of object coefficients of restitutia.
% G               Strength of gravity / Gravitational force constant 6.67 * 10^-11
% dt              Time step, in  Earth years
% N               Number of time steps

function [ x, y, z, vx, vy, vz, ax, ay, az, collides ] =...
    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, C, G, dt, N )

%Number of bodies
B = length(x0);

%Initialise output arrays
x = repmat( x0, [N,1] );
y = repmat( y0, [N,1] );
z = repmat( z0, [N,1] );
vx = repmat( vx0, [N,1] );
vy = repmat( vy0, [N,1] );
vz = repmat( vz0, [N,1] );
collides = zeros(N,B);

%Work out inital acceleration
[ax0,ay0,az0] = newton( x0,y0,z0,G,M );
ax = repmat( ax0, [N,1] );
ay = repmat( ay0, [N,1] );
az = repmat( az0, [N,1] );

%Work out subsequent dynamics of all bodies via Verlet + Newtonian Gravity
for n=2:N
    
    %Update positions
    x(n,:) = x(n-1,:) + dt * vx(n-1,:) + 0.5*dt*dt*ax(n-1,:);
    y(n,:) = y(n-1,:) + dt * vy(n-1,:) + 0.5*dt*dt*ay(n-1,:);
    z(n,:) = z(n-1,:) + dt * vz(n-1,:) + 0.5*dt*dt*az(n-1,:);
    
    %Update acceleration
    [Ax,Ay,Az] = newton( x(n,:),y(n,:),z(n,:),G,M );
    ax(n,:) = Ax(:);
    ay(n,:) = Ay(:);
    az(n,:) = Az(:);
    
    %Determine whether any non-zero masses are colliding, and compute
    %outcome if they are deemed to collide
    [collides(n,:), vx(n,:),vy(n,:),vz(n,:),vmax,...
        KE_pre,KE_post,KE_tot_pre,KE_tot_post ] =...
        bounce( x(n,:),y(n,:),z(n,:),vx(n,:),vy(n,:),vz(n,:),M,R,C );
    
    %Set acceleration to zero for colliding objects
    b = find( collides(n,:)~=0 );
    ax(n,b) = 0;
    ay(n,b) = 0;
    az(n,b) = 0;   
    
    %Update velocity of objects that are not colliding using acceleration
    nb = find( collides(n,:)==0 );
    vx(n,nb) = vx(n-1,nb) + 0.5*dt*( ax(n,nb) + ax(n-1,nb) );
    vy(n,nb) = vy(n-1,nb) + 0.5*dt*( ay(n,nb) + ay(n-1,nb) );
    vz(n,nb) = vz(n-1,nb) + 0.5*dt*( az(n,nb) + az(n-1,nb) );
end

%%

%Compute acceleration of all objects via Newtonian gravity
% Note x,y,z are in units of AU, M are in Earth masses
function [ax,ay,az] = newton( x,y,z,g,M )

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_sun = 2e30;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_sun/(AU^3);

%Number of bodies
B = length(M);

%Work out bodies which don't have zero mass
nzm = find(M~=0);
zm = find(M==0);

%Initialise accelerations
ax = zeros(1,B);
ay = zeros(1,B);
az = zeros(1,B);

%

%Compute acceleration of non-zero masses

%Compute accelerations resulting from each mass and store these in a matrix
%aa.
d = distance( [x(nzm);y(nzm);z(nzm)],[x(nzm);y(nzm);z(nzm)] );
m = repmat( M(nzm).',[1,length(nzm)] );

%x component of acceleration
aa = m.*( repmat( x(nzm).',[1,length(nzm)] ) - repmat( x(nzm),[length(nzm),1] ))./d.^3;
aa(1:length(nzm)+1:end) = 0;  %Don't let self same mass displacements contribute!
ax(nzm) = P*g*sum(aa,1);

%y component of acceleration
aa = m.*( repmat( y(nzm).',[1,length(nzm)] ) - repmat( y(nzm),[length(nzm),1] ))./d.^3;
aa(1:length(nzm)+1:end) = 0; %Don't let self same mass displacements contribute!
ay(nzm) = P*g*sum(aa,1);

%z component of acceleration
aa = m.*( repmat( z(nzm).',[1,length(nzm)] ) - repmat( z(nzm),[length(nzm),1] ))./d.^3;
aa(1:length(nzm)+1:end) = 0; %Don't let self same mass displacements contribute!
az(nzm) = P*g*sum(aa,1);

%

%Compute acceleration of zero masses

%Compute accelerations resulting from each mass and store these in a matrix
%aa.
d = distance( [x(nzm);y(nzm);z(nzm)],[x(zm);y(zm);z(zm)] );
m = repmat( M(nzm).',[1,length(zm)] );

%x component of acceleration
aa = m.*( repmat( x(nzm).',[1,length(zm)] ) - repmat( x(zm),[length(nzm),1] ))./d.^3;
ax(zm) = P*g*sum(aa,1);

%y component of acceleration
aa = m.*( repmat( y(nzm).',[1,length(zm)] ) - repmat( y(zm),[length(nzm),1] ))./d.^3;
ay(zm) = P*g*sum(aa,1);

%z component of acceleration
aa = m.*( repmat( z(nzm).',[1,length(zm)] ) - repmat( z(zm),[length(nzm),1] ))./d.^3;
az(zm) = P*g*sum(aa,1);

%End of code