%collide
% For a two body collision, colide.m computes resultant velocities and 
% kinetic energies. Note all inputs can be matrices, but these must be the
% same dimensions.
%
% LAST UPDATED by Andy French May 2012.
%
% Syntax: [vx1,vy1,vz1,...
%     vx2,vy2,vz2,...
%     vmax,...
%     KE1_pre,KE1_post,...
%     KE2_pre,KE_post ] = collide( ux1,uy1,uz1, ux2,uy2,uz2, m1, m2, C )
%
% ux1,uy1,uz1   Initial velocities of mass 1 /ms^-1
% ux2,uy2,uz2   Initial velocities of mass 2 /ms^-1
% m1,m2         Colliding masses /kg
% C             Coefficient of restitution. 
%               ( = Speed of separation / speed of approach)
%               C=1 => Elastic
%               C=0 => Inelastic
%
% vx1,vy1,vz1   Velocities following collision of mass 1 /ms^-1
% vx2,vy2,vz2   Velocities following collision of mass 2 /ms^-1
% vmax          Maximum velocity /ms^-1
% KE...         Kinetic energy /J. 
%              (Mass 1,2 and total pre and post collision)

function [vx1,vy1,vz1,...
    vx2,vy2,vz2,...
    vmax,...
    KE1_pre,KE1_post,...
    KE2_pre,KE2_post,...
    KE_tot_pre,KE_tot_post] = collide( ux1,uy1,uz1, ux2,uy2,uz2, m1, m2, C )

%Compute post collision velocities

%Mass 1
vx1 = ux1.* ( m1.*(1+C)./(m1+m2) - C ) + ux2.*m2.*(C+1)./(m1+m2) ;
vy1 = uy1.* ( m1.*(1+C)./(m1+m2) - C ) + uy2.*m2.*(C+1)./(m1+m2) ;
vz1 = uz1.* ( m1.*(1+C)./(m1+m2) - C ) + uz2.*m2.*(C+1)./(m1+m2) ;

%Mass 2
vx2 = ux1.*m1.*(1+C)./(m1+m2) + ux2.*( m2.*(C+1)./(m1+m2) - C );
vy2 = uy1.*m1.*(1+C)./(m1+m2) + uy2.*( m2.*(C+1)./(m1+m2) - C );
vz2 = uz1.*m1.*(1+C)./(m1+m2) + uz2.*( m2.*(C+1)./(m1+m2) - C );

%Find maximum velocity
v = [vx1,vy1,vz1,vx2,vy2,vz2,ux1,uy1,uz1,ux2,uy2,uz2];
v = reshape( v, [1,numel(v)] );
vmax = max(v);

%Compute pre and post collision kinetic energies
KE1_pre = 0.5*m1.*sqrt( ux1.^2 + uy1.^2 + uz1.^2 );
KE2_pre = 0.5*m2.*sqrt( ux2.^2 + uy2.^2 + uz2.^2 );
KE1_post = 0.5*m1.*sqrt( vx1.^2 + vy1.^2 + vz1.^2 );
KE2_post = 0.5*m2.*sqrt( vx2.^2 + vy2.^2 + vz2.^2 );
KE_tot_pre = KE1_pre + KE2_pre;
KE_tot_post = KE1_post + KE2_post;

%End of code

