%bounce,
% Function which updates the velocity x,y,z vectors of N objects, each with
% 'collision radii' R, after checking for the possibility of a collision.
% If a collision is determined, the first pair is chosen from those
% involved with each collision and their velocities are updated according
% conservation of momentum and the coefficient of restitution C between the
% bodies.
%
% Input positions x,y,z and velocities ux,uy,uz are row vectors.
%
% LAST UPDATED by Andy French May 2012
%
% Syntax: [collisions, vx,vy,vz,vmax, KE_pre,KE_post,KE_tot_pre,KE_tot_post ] =...
%    bounce( x,y,z,ux,uy,uz,m,R,C )
%
% x,y,z         Vector of object positions /m
% ux,uy,uz      Vector of initial velocities  /ms^-1
% m             Vector of (potentially) colliding masses /kg
% C             Coefficient of restitution. 
%               ( = Speed of separation / speed of approach)
%               C=1 => Elastic
%               C=0 => Inelastic
%               Note this must be a square matrix, with rows = number of
%               objects. i.e. C(4,5) is the coefficient of restitution
%               between objects 4 and 5. Clearly C(i,j) = C(j,i).
% R             Vector of collision radii. If objects (i,j) approach within
%               R(i) + R(j) then they are said to have collided.
%
% vx,vy,vz      Velocities following collision of mass 1 /ms^-1
% vmax          Maximum velocity /ms^-1
% KE...         Kinetic energy /J. 
% collides      Vector which states which object is colliding with the
%               column number. 0 implies no collision.

function [collides, vx,vy,vz,vmax, KE_pre,KE_post,KE_tot_pre,KE_tot_post ] =...
    bounce( x,y,z,ux,uy,uz,m,R,C )

%Work out bodies which don't have zero mass
nzm = find(m~=0);
zm = find(m==0);

%Initialize collides vector, which describes which elements are colliding
%with which others. Note N-body collisions are allreduced to two.
collides = zeros(1,length(x));

%Initialize output velocity vectors
vx = ux;
vy = uy;
vz = uz;

%Collisions with non-zero masses and zero masses
if ~isempty(zm)
[collides(zm),vx(zm),vy(zm),vz(zm)] = ...
    nbodybounce( x(nzm),x(zm),y(nzm),y(zm),z(nzm),z(zm),...
    ux(nzm),ux(zm),uy(nzm),uy(zm),uz(nzm),uz(zm),R(nzm),...
    R(zm),m(nzm),m(zm),C(nzm,zm) );
end

%Collisions with non-zero masses and non-zero masses
if ~isempty(nzm)
[collides(nzm), vx(nzm),vy(nzm),vz(nzm)] = ...
    nbodybounce( x(nzm),x(nzm),y(nzm),y(nzm),z(nzm),z(nzm),...
    ux(nzm),ux(nzm),uy(nzm),uy(nzm),uz(nzm),uz(nzm),R(nzm),...
    R(nzm),m(nzm),m(nzm),C(nzm,nzm) );
end

%Note we don't care about the collisions between zero masses!

%Compute vector of initial kinetic energies
KE_pre = 0.5 * m.*sqrt( ux.^2 + uy.^2 + uz.^2 );
KE_tot_pre = sum( KE_pre );

%Determine maximum velocity
v = sqrt( vx.^2 + vy.^2 + vz.^2 );
vmax = max(v);

%Compute vector of final kinetic energies
KE_post = 0.5 * m.*v.^2;
KE_tot_post = sum( KE_post );

%%

%N-body collision function. Constructs matrix of size length(x1) x
%length(x2) of inter-object distances and compares these to the sum of
%collision radii. Note coefficient of restitution matrix C has dimensions
% [length(x1),length(x2)]
function [collides,vx2,vy2,vz2] = ...
    nbodybounce( x1,x2,y1,y2,z1,z2,ux1,ux2,uy1,uy2,uz1,uz2,R1,R2,m1,m2,C )

%Construct matrix of inter-object distances
d = distance( [x1;y1;z1],[x2;y2;z2] );

%Use initial velocities to determine whether objects are actually
%approaching each other.

%%%% TBD %%%%

%Prevent objects getting trapped, when the recoil velocity is insufficient
%to escape by the combined object radii in the next time step.

%%%% TBD %%%%

%Compute equivalent matrix to d corresponding to the sum of the object
%collision radii
Rsum = repmat( R2,[length(R1),1] ) + repmat( R1.',[1,length(R2)] );

%Compute binary array which indicates collisions
collisions = d<Rsum;  % x logic which determines whether objects are indeed approaching each other

%Set block diagonal (self collisions) to be zero. i.e. not a collision!
collisions( 1:length(R2)+1:end ) = 0;

%If a column has more than one 1 (i.e. a multi-body collision), chose a row
%number at random to be 1 and set the others to be zero.
collisions_sum = sum( collisions,1 );
n_body_collision = find( collisions_sum > 1 );
if ~isempty(n_body_collision)
    for n=1:length(n_body_collision)
        c = find( collisions(:,n_body_collision(n) ));
        collisions( :,n_body_collision(n) ) = 0;
        collisions(c(1),n_body_collision(n)) = 1;
    end
end

%Determine object indices involved with a collision
[obj1,obj2] = find( collisions == 1 );
obj1 = obj1.';
obj2 = obj2.';

%Modify velocities of these objects as a result of collision, conserving
%momentum.
vx1 = ux1;
vy1 = uy1;
vz1 = uz1;
vx2 = ux2;
vy2 = uy2;
vz2 = uz2;
[vx1(obj1),vy1(obj1),vz1(obj1),...
    vx2(obj2),vy2(obj2),vz2(obj2),...
    vmax,...
    KE1_pre,KE1_post,...
    KE2_pre,KE2_post,...
    KE_tot_pre,KE_tot_post] = collide( ux1(obj1),uy1(obj1),uz1(obj1),...
    ux2(obj2),uy2(obj2),uz2(obj2), m1(obj1), m2(obj2), C(sub2ind(size(C),obj1,obj2)) );

%Construct output collisions vector
collides = zeros(1,length(R2));
collides(obj1) = obj2;

%%

function bounce_test

%Positions
x = [1,2,1.1];
y = [-1,2,-1.1];
z = [0,0,0];

%Velocities
ux = [1,-1,1];
uy = [0,-1,1];
uz = [0,0,0];

%Masses
m = [1,2,3];

%Collision radii
R = [0.5,0.5,0.5];

%Coefficient of restitution
C = rand(3,3);

%Compute result of any collisions
[collides, vx,vy,vz,vmax, KE_pre,KE_post,KE_tot_pre,KE_tor_post ] =...
    bounce( x,y,z,ux,uy,uz,m,R,C )

%End of code