% two_stars_globular_gravity_sim
%
% LAST UPDATED by Andy French May 2012.

function two_stars_globular_gravity_sim

%%% INPUTS %%%

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_earth = 5.97e24;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_earth/(AU^3);

%Masses in kg of Earth and stars 1 and 2
M_earth = 5.97e24;
M1 = 2e30;
M2 = 1*M1;

%Star separation /AU
s = 2;

%Massless planet parameters
num_masses_per_square_AU = 5;
num_shells = 10;
shell_separation_AU = 0.2;
r0 = 1; %Radius in AU from star that rings start

%Strength of gravity
G = 1;

%Timestep / years
dt = 0.01;

%

%Set graph limits /AU
AUmax = 8;

%Set maximum orbital velocity for plot colouring
vmax = 2;

%Marker size for planets
msize = 2;

%Set view (2 or 3)
view_option = 3;

%%

%Compute initial orbital parameters for the two stars such that they rotate
%in a circular fashion about their mutual centre of mass
[vx1,vx2,vy1,vy2,vz1,vz2,x1,y1,z1,x2,y2,z2,xx,yy,zz] =...
    two_body_circular_init( M1/M_earth, M2/M_earth, s );

%Change z orientation for strange effects!
vz1 = vy1; vy1 = 0;
vz2 = vy2; vy2 = 0;
vz1=0;
vz2=0;

%Planets starting in concentric rings about star
[xx1,yy1,zz1,vxx1,vyy1,vzz1,num_masses1] = globular_cluster( x1,y1,z1,M1/M_earth,shell_separation_AU,...
    num_shells, r0, num_masses_per_square_AU, P);

%Planets starting in concentric rings about star 2
[xx2,yy2,zz2,vxx2,vyy2,vzz2,num_masses2] = globular_cluster( x2,y2,z2, M2/M_earth,shell_separation_AU,...
    num_shells, r0, num_masses_per_square_AU, P);

%Assemble position and velocity vectors for all objects
x0 = [x1,x2,xx1,xx2];
y0 = [y1,y2,yy1,yy2];
z0 = [z1,z2,zz1,zz2];
vx0 = [vx1,vx2,vxx1,vxx2];
vy0 = [vy1,vy2,vyy1,vyy2];
vz0 = [vz1,vz2,vzz1,vzz2];

%Radii and masses
R = [0.5,0.001,zeros(1,num_masses1),zeros(1,num_masses2)];
M = [M1/M_earth,M2/M_earth,zeros(1,num_masses1),zeros(1,num_masses2),];

%Coefficients of restitutia
C = 0.1*ones(length(R),length(R));

%%

%% GRAVITY SIM ANIMATION %%

%Set up figure
fig = figure('color',[1 1 1],'name','gravity simulation animation',...
    'renderer','painters','KeyPressFcn',@keypressfunc,'units','normalized','position',[0 0 1 1]);
d = get(fig,'userdata');
d.run = 1;
d.save = 0;
d.print = 0;
d.wait = 0;
t = 0;
set(fig,'userdata',d);

%Set up axis
cmap = colormap;
colorbar
caxis([0,vmax])
hold on;
xlabel('x /AU')
ylabel('y /AU')
zlabel('z /AU')
tit = title(['Gravity sim: M_1/M_2 = ',num2str(M1/M2),'. t = ',num2str(t),' years.']);
view(view_option)
axis vis3d
axis equal
xlim([-AUmax,AUmax])
ylim([-AUmax,AUmax])
zlim([-AUmax,AUmax])
grid on;
hold on;

%Initialise object data vectors
x = x0;
y = y0;
z = z0;
vx = vx0;
vy = vy0;
vz = vz0;

%Plot initial positions. Colour code two stars by velocity
[r,g,b] = x_to_color(sqrt(vx(1:2).^2 + vy(1:2).^2 + vz(1:2).^2 ),cmap,0,vmax);
p1 = plot3(x0(1),y0(1),z0(1),'ro','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p2 = plot3(x0(2),y0(2),z0(2),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p3 = plot3(x0(3:3+num_masses1-1),y0(3:3+num_masses1-1),z0(3:3+num_masses1-1),'r.','markersize',msize);
p4 = plot3(x0(3+num_masses1:end),y0(3+num_masses1:end),z0(3+num_masses1:end),'b.','markersize',msize);

%Run simulation until user presses 'q'
while d.run == 1
    %Obtain figure user data and check for key presses
    d = get(fig,'userdata');   
    if d.wait == 1
        pause(0.1);
    elseif d.wait == 0
        
        %Update positions, veocities and accelerations and times
        [ x, y, z, vx, vy, vz, ax, ay, az, collisions ] =...
            gravity( x, y, z, vx, vy, vz, R, M, C, G, dt, 2 );
        t = t + dt;
        x = x(2,:);
        y = y(2,:);
        z = z(2,:);
        vx = vx(2,:);
        vy = vy(2,:);
        vz = vz(2,:);
        
        %Update plot. Leave a persistent trail of star mass positions
        [r,g,b] = x_to_color( sqrt(vx(1:2).^2 + vy(1:2).^2 + vz(1:2).^2 ),cmap,0,vmax);
        set(p1,'Xdata',x(1),'Ydata',y(1),'Zdata',z(1),'markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
        set(p2,'Xdata',x(2),'Ydata',y(2),'Zdata',z(2),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        plot3(x(1),y(1),z(1),'r.','markersize',1);
        plot3(x(2),y(2),z(2),'b.','markersize',1);
        set(p3,'Xdata',x(3:3+num_masses1-1),'Ydata',y(3:3+num_masses1-1),'Zdata',z(3:3+num_masses1-1));
        set(p4,'Xdata',x(3+num_masses1:end),'Ydata',y(3+num_masses1:end),'Zdata',z(3+num_masses1:end));
        
        %Update title
        set(tit,'string',['Gravity sim: M_1/M_2 = ',num2str(M1/M2),'. t = ',num2str(t),' years.'])
        
        %Flush pending graphics requests
        drawnow
        
        %Special actions resulting from key presses
        d = get(fig,'userdata');
        if d.save == 1
            %Save a .mat file of the current data set
            save( ['gravity_sim_data t=',num2str(t),'.mat'] );
            d.save = 0;
        end
        if d.print == 1;
            %Print a screenshot of the current view
            print( fig, '-dpng','-r300',['two star globular gravity sim. t=',num2str(t),...
                '.  ',strrep(datestr(now),':','-'),'.png'])
            d.print = 0;
        end
        set(fig,'userdata',d);
    end
end
close(fig);

%%

%Figure key press function callback
function keypressfunc( fig,evnt )
d = get(fig,'userdata');
if evnt.Character == 'q'
    d.run = 0;
elseif evnt.Character == 'p'
    d.print = 1;
elseif evnt.Character == 's'
    d.save = 1;
elseif evnt.Character == 'w'
    d.wait = 1;
elseif evnt.Character == 'c'
    d.wait = 0;
end
set(fig,'userdata',d);

%End of code