%two_body_circular_init
% Function which returns initial conditions for mutually circular orbits 
% given masses m1,m2 (in Earth masses) and separation s in AU. 
% Initial x,y,z coordinates are in AU and velocities are in AU per Year.
%
% The centre of mass of the system is defined to be the origin of the
% coordinate systems.
%
% LAST UPDATED by Andy French May 2012.

function [vx1,vx2,vy1,vy2,vz1,vz2,x1,y1,z1,x2,y2,z2,xx,yy,zz] =...
    two_body_circular_init( m1, m2, s )

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_earth = 5.97e24;
Yr = 365*24*3600;

%Dimensionless number which controls the gravitational dynamics, and results from the
%scaling of mass, distance and time parameters.
P = G*Yr^2*M_earth/(AU^3);

%Define initial position coordinates
x1 = 0;
x2 = s;
y1 = 0;
y2 = 0;
z1 = 0;
z2 = 0;

%Compute centre of mass coordinates
xx = ( m1*x1 + m2*x2 )/(m1+m2);
yy = ( m1*y1 + m2*y2 )/(m1+m2);
zz = ( m1*z1 + m2*z2 )/(m1+m2);

%Shift coordinate system so centre of mass is the origin
x1 = x1 - xx;
x2 = x2 - xx;
y1 = y1 - yy;
y2 = y2 - yy;
z1 = z1 - zz;
z2 = z2 - zz;

%Define initial velocity coordinates to result in circular orbits in the
%x,y plane
vx1 = 0;
vx2 = 0;
vy1 = sqrt( P * (m1 + m2) * xx^2 / s^3 );
vy2 = -sqrt( P * (m1 + m2) * (s-xx)^2 / s^3 );
vz1 = 0;
vz2 = 0;

%End of code