% solar
%
% LAST UPDATED by us.

function solar_system_gravity_sim

%%% INPUTS %%%

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_earth = 5.97e24;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_earth/(AU^3);

%Masses in Earth units
M_earth = 1;
M_sun = 2e30 / 5.97e24;
M_mercury = 0.055;
M_venus = 0.815;
M_mars = 0.107;
M_jupiter = 317.8;
M_saturn = 95.152;
M_uranus = 1e4*14.536;
M_neptune = 17.147;

%Star separation /AU
s_earth = 1;
s_mercury = 0.387;
s_venus = 0.723;
s_mars = 1.523;
s_jupiter = 5.204;
s_saturn = 9.582;
s_uranus = 19.229;
s_neptune = 30.103;

%Massless planet parameters
arc_separation_AU = 2*pi/30;
num_rings = 20;
ring_radius_diff_AU = 0.5;
r0 = 5; %Radius in AU from star that rings start

%Strength of gravity
G = 1;

%Timestep / years
dt = 0.01;

%

%Set graph limits /AU
AUmax = 15;

%Set maximum orbital velocity for plot colouring
vmax = 2;

%Marker size for planets
msize = 2;

%Set view (2 or 3)
view_option = 3;

%%

%Compute initial orbital parameters for the two stars such that they rotate
%in a circular fashion about their mutual centre of mass
[vx1,vx2,vy1,vy2,vz1,vz2,x1,y1,z1,x2,y2,z2,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_mercury, s_mercury );

[vx1,vx3,vy1,vy3,vz1,vz3,x1,y1,z1,x3,y3,z3,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_venus, s_venus );

[vx1,vx4,vy1,vy4,vz1,vz4,x1,y1,z1,x4,y4,z4,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_earth, s_earth );

[vx1,vx5,vy1,vy5,vz1,vz5,x1,y1,z1,x5,y5,z5,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_mars, s_mars );

[vx1,vx6,vy1,vy6,vz1,vz6,x1,y1,z1,x6,y6,z6,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_jupiter, s_jupiter );

[vx1,vx7,vy1,vy7,vz1,vz7,x1,y1,z1,x7,y7,z7,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_saturn, s_saturn );

[vx1,vx8,vy1,vy8,vz1,vz8,x1,y1,z1,x8,y8,z8,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_uranus, s_uranus );

[vx1,vx9,vy1,vy9,vz1,vz9,x1,y1,z1,x9,y9,z9,xx,yy,zz] =...
    two_body_circular_init( M_sun, M_neptune, s_neptune );


%Planets starting in concentric rings about star
[xx1,yy1,zz1,vxx1,vyy1,vzz1,num_masses1] = mass_rings( x1,y1,z1, M_sun, arc_separation_AU,...
    num_rings, r0, ring_radius_diff_AU, P);

%Assemble position and velocity vectors for all objects
x0 = [x1,x2,x3,x4,x5,x6,x7,x8,x9,xx1];
y0 = [y1,y2,y3,y4,y5,y6,y7,y8,y9,yy1];
z0 = [z1,z2,z3,z4,z5,z6,z7,z8,z9,zz1];
vx0 = [vx1,vx2,vx3,vx4,vx5,vx6,vx7,vx8,vx9,vxx1];
vy0 = [vy1,vy2,vy3,vy4,vy5,vy6,vy7,vy8,vy9,vyy1];
vz0 = [vz1,vz2,vz3,vz4,vz5,vz6,vz7,vz8,vz9,vzz1];

%Radii and masses
R = [0.01,0.001,0.001,0.001,0.001,0.001,0.001,0.001,0.001,0.001*ones(1,num_masses1)];
M = [M_sun,M_mercury,M_venus,M_earth,M_mars,M_jupiter,M_saturn,M_uranus,M_neptune,zeros(1,num_masses1)];

%Coefficients of restitutia
C = ones(length(R),length(R));

%%

%% GRAVITY SIM ANIMATION %%

%Set up figure
fig = figure('color',[1 1 1],'name','gravity simulation animation',...
    'renderer','painters','KeyPressFcn',@keypressfunc,'units','normalized','position',[0 0 1 1]);
d = get(fig,'userdata');
d.run = 1;
d.save = 0;
d.print = 0;
d.wait = 0;
t = 0;
set(fig,'userdata',d);
tit = title(['Gravity sim: M_sun = ',num2str(M_sun),'. t = ',num2str(t),' years.']);

%Set up axis
cmap = colormap;
colorbar
caxis([0,vmax])
hold on;
xlabel('x /AU')
ylabel('y /AU')
zlabel('z /AU')
view(view_option)
axis vis3d
axis equal
xlim([-AUmax,AUmax])
ylim([-AUmax,AUmax])
zlim([-AUmax,AUmax])
grid on;
hold on;

%Initialise object data vectors
x = x0;
y = y0;
z = z0;
vx = vx0;
vy = vy0;
vz = vz0;

%Plot initial positions. Colour code two stars by velocity
[r,g,b] = x_to_color(sqrt(vx(1:2).^2 + vy(1:2).^2 + vz(1:2).^2 ),cmap,0,vmax);
p1 = plot3(x0(1),y0(1),z0(1),'ro','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)],'marker','*','markersize',30);
p2 = plot3(x0(2),y0(2),z0(2),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p3 = plot3(x0(3),y0(3),z0(3),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p4 = plot3(x0(4),y0(4),z0(4),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p5 = plot3(x0(5),y0(5),z0(5),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p6 = plot3(x0(6),y0(6),z0(6),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p7 = plot3(x0(7),y0(7),z0(7),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p8 = plot3(x0(8),y0(8),z0(8),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p9 = plot3(x0(9),y0(9),z0(9),'bo','markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
p10 = plot3(x0(10:10+num_masses1-1),y0(10:10+num_masses1-1),z0(10:10+num_masses1-1),'r.','markersize',msize);


%Run simulation until user presses 'q'
while d.run == 1
    %Obtain figure user data and check for key presses
    d = get(fig,'userdata');
    if d.wait == 1
        pause(0.1);
    elseif d.wait == 0
        
        %Update positions, veocities and accelerations and times
        [ x, y, z, vx, vy, vz, ax, ay, az, collisions ] =...
            gravity( x, y, z, vx, vy, vz, R, M, C, G, dt, 2 );
        t = t + dt;
        x = x(2,:);
        y = y(2,:);
        z = z(2,:);
        vx = vx(2,:);
        vy = vy(2,:);
        vz = vz(2,:);
        
        %Update plot. Leave a persistent trail of star mass positions
        [r,g,b] = x_to_color( sqrt(vx(1:2).^2 + vy(1:2).^2 + vz(1:2).^2 ),cmap,0,vmax);
        set(p1,'Xdata',x(1),'Ydata',y(1),'Zdata',z(1),'markerfacecolor',[r(1),g(1),b(1)],'markeredgecolor',[r(1),g(1),b(1)]);
        set(p2,'Xdata',x(2),'Ydata',y(2),'Zdata',z(2),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p3,'Xdata',x(3),'Ydata',y(3),'Zdata',z(3),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p4,'Xdata',x(4),'Ydata',y(4),'Zdata',z(4),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p5,'Xdata',x(5),'Ydata',y(5),'Zdata',z(5),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p6,'Xdata',x(6),'Ydata',y(6),'Zdata',z(6),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p7,'Xdata',x(7),'Ydata',y(7),'Zdata',z(7),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p8,'Xdata',x(8),'Ydata',y(8),'Zdata',z(8),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        set(p9,'Xdata',x(9),'Ydata',y(9),'Zdata',z(9),'markerfacecolor',[r(2),g(2),b(2)],'markeredgecolor',[r(2),g(2),b(2)]);
        plot3(x(1),y(1),z(1),'r.','markersize',1);
        plot3(x(2),y(2),z(2),'b.','markersize',1);
        plot3(x(3),y(3),z(3),'b.','markersize',1);
        plot3(x(4),y(4),z(4),'b.','markersize',1);
        plot3(x(5),y(5),z(5),'b.','markersize',1);
        plot3(x(6),y(6),z(6),'b.','markersize',1);
        plot3(x(7),y(7),z(7),'b.','markersize',1);
        plot3(x(8),y(8),z(8),'b.','markersize',1);
        plot3(x(9),y(9),z(9),'b.','markersize',1);
        set(p10,'Xdata',x(10:10+num_masses1-1),'Ydata',y(10:10+num_masses1-1),'Zdata',z(10:10+num_masses1-1));
        
        %Update title
        set(tit,'string',['Gravity sim: M_sun = ',num2str(M_sun),'. t = ',num2str(t),' years.'])
        
        %Flush pending graphics requests
        drawnow
        
        %Special actions resulting from key presses
        d = get(fig,'userdata');
        if d.save == 1
            %Save a .mat file of the current data set
            save( ['gravity_sim_data t=',num2str(t),'.mat'] );
            d.save = 0;
        end
        if d.print == 1;
            %Print a screenshot of the current view
            print( fig, '-dpng','-r300',['Solar system gravity sim. t=',num2str(t),...
                '.  ',strrep(datestr(now),':','-'),'.png'])
            d.print = 0;
        end
        set(fig,'userdata',d);
    end
end
close(fig);

%%

%Figure key press function callback
function keypressfunc( fig,evnt )
d = get(fig,'userdata');
if evnt.Character == 'q'
    d.run = 0;
elseif evnt.Character == 'p'
    d.print = 1;
elseif evnt.Character == 's'
    d.save = 1;
elseif evnt.Character == 'w'
    d.wait = 1;
elseif evnt.Character == 'c'
    d.wait = 0;
end
set(fig,'userdata',d);

%End of code