%Compute acceleration of all objects via Newtonian gravity
% Note x,y,z are in units of AU, M are in Earth masses
function [ax,ay,az] = newton( x,y,z,g,M,R,k )

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_earth = 5.97e24;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_earth/(AU^3);

%Number of bodies
B = length(M);

%Work out bodies which don't have zero mass
nzm = find(M~=0);
zm = find(M==0);

%Initialise accelerations
ax = zeros(1,B);
ay = zeros(1,B);
az = zeros(1,B);

%Compute accelerations for non-zero masses
for j=1:length(nzm)
    for jj = 1:length(nzm)
        %Ignore self reference to body b!
        if jj~=j
            
            %Find body index from array of body indices which
            %correspond to non zero masses
            b = nzm(j);
            bb = nzm(jj);
            
            %Compute inter-mass distance
            d = ( ( x(bb) - x(b) )^2 + ( y(bb) - y(b) )^2 + ( z(bb) - z(b))^2 )^(1/2);
            
            %Newton's Law of gravitation, unless r_cubed is less than sum
            %of object radii. In this case we will treat the objects as
            %elastic bodies colliding.
            if d > k*( R(b) + R(bb) )
                ax(b) = ax(b) + P*g * M(bb) * ( x(bb) - x(b) ) / d^3;
                ay(b) = ay(b) + P*g * M(bb) * ( y(bb) - y(b) ) / d^3;
                az(b) = az(b) + P*g * M(bb) * ( z(bb) - z(b) ) / d^3;
            else
                ax(b) = 0;
                ay(b) = 0;
                az(b) = 0;
            end
        end
    end
end

%Compute accelerations for zero masses. Only consider gravitational forces
%resulting from non-zero masses.
for j = 1:length(nzm)
    
    %Find body index from array of body indices which
    %correspond to non zero masses
    b = nzm(j);
    
    %Compute inter-mass distance
    d = ( ( x(b) - x ).^2 + ( y(b) - y ).^2 + ( z(b) - z ).^2 ).^(1/2);
    
    %Newton's Law of gravitation, unless r_cubed is less than sum
    %of object radii. In this case we will treat the objects as
    %elastic bodies colliding, and don't modify acceleration.
    c = find( d < k*( R + R(b) ));
    c = setdiff( c, nzm );
    nc = find( d >= k*( R + R(b) ));
    nc = setdiff( nc, nzm );
    ax(nc) = ax(nc) + P*g * M(b) * ( x(b) - x(nc) ) ./ d(nc).^3;
    ay(nc) = ay(nc) + P*g * M(b) * ( y(b) - y(nc) ) ./ d(nc).^3;
    az(nc) = az(nc) + P*g * M(b) * ( z(b) - z(nc) ) ./ d(nc).^3;
    ax(c) = 0;
    ay(c) = 0;
    az(c) = 0;
end

%%

%Compute gravitational potential at coordinates given in equal sized matrices xx,yy and zz
%based upon mass coordinates in vectors x,y,z. g is the multiplier of the
%gravitational force constant G and M is a vector os masses /earth masses.
%All spatial dimensions are in AU. Note zero masses do not contribute to
%the potential.
function phi = grav_potential( xx,yy,zz,x,y,z,g,M )
phi = zeros(size(xx));
nzm = find(M~=0);
for n=1:length(nzm)
    m = nzm(n);
    phi = phi + g*M(m)./ sqrt( ( xx - x(m)).^2 +...
        ( yy - y(m)).^2 + ( zz - z(m)).^2 );
end
phi = -phi;

%End of code