%gravity
% N-body gravity simulator. Uses the Verlet algorithm + Newtonian gravity
% to determine future x,y,z coordinates (and velocities and accelerations)
% of N masses.
%
% [ t, x, y, z, vx, vy, vz, ax, ay, az ] =...
%    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, G, k, dt, N, bodies )
%
% t               Vector of simulation times / Earth years
% x,y,z           x,y,z coordinates in AU (distance between the Earth and
%                 the Sun
% vx,vy,vz        Cartesian velocities in AU per Earth year
% ax,ay,az        Cartesian accelerations in AU per square Earth years.
% bounce          Boolean flag. 1 indicates body X is in a 'bounce
%                 condition' or collision state.
% x0,y0,z0        Initial object positions in AU
% vx0,vy0,vz0     Initial ibject velocities in AU per Earth year
% R               Row vector of object raddi / Earth radii
% M               Row vector of object masses / Earth masses
% G               Strength of gravity / Gravitational force constant 6.67 * 10^-11
% k               k * ( sum of colliding planet radii ) is 'gravity
%                 coasting' separation (or 'bounce condition')
% dt              Time step, in  Earth years
% N               Number of time steps

function [ x, y, z, vx, vy, vz, ax, ay, az, bounce ] =...
    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, G, k, dt, N )

%Number of bodies
B = length(x0);

%Initialise output arrays
x = repmat( x0, [N,1] );
y = repmat( y0, [N,1] );
z = repmat( z0, [N,1] );
vx = repmat( vx0, [N,1] );
vy = repmat( vy0, [N,1] );
vz = repmat( vz0, [N,1] );
bounce = zeros(N,B);

%Work out inital acceleration
[ax0,ay0,az0] = newton( x0,y0,z0,G,M,R,k );
ax = repmat( ax0, [N,1] );
ay = repmat( ay0, [N,1] );
az = repmat( az0, [N,1] );

%Work out subsequent dynamics of all bodies via Verlet + Newtonian Gravity
for n=2:N
    
    %Update positions
    for b=1:B
        x(n,b) = x(n-1,b) + dt * vx(n-1,b) + 0.5*dt*dt*ax(n-1,b);
        y(n,b) = y(n-1,b) + dt * vy(n-1,b) + 0.5*dt*dt*ay(n-1,b);
        z(n,b) = z(n-1,b) + dt * vz(n-1,b) + 0.5*dt*dt*az(n-1,b);
    end
    
    %Check for (elastic) bounce conditions.
    for b=1:B
        for bb=1:B
            %Ignore self reference to body b!
            if bb~=b
                %Compute inter-mass distance
                d = ( ( x(n,bb) - x(n,b) )^2 + ( y(n,bb) - y(n,b) )^2 + ( z(n,bb) - z(n,b))^2 )^(1/2);
                if d <= k*( R(b) + R(bb) )
                    bounce(n,bb) = 1;
                end
            end
        end
    end
    
    %Update acceleration
    for b=1:B
        [Ax,Ay,Az] = newton( x(n,:),y(n,:),z(n,:),G,M,R,k );
        ax(n,b) = Ax(b);
        ay(n,b) = Ay(b);
        az(n,b) = Az(b);
    end
    
    %Update velocity
    for b=1:B
        if bounce(n,b)==1
            %Elastic collision
            vx(n,b) = -vx(n-1,b);
            vy(n,b) = -vy(n-1,b) ;
            vz(n,b) = -vz(n-1,b) ;
        else
            %Use Newtonian gravity to change velocity
            vx(n,b) = vx(n-1,b) + 0.5*dt*( ax(n,b) + ax(n-1,b) );
            vy(n,b) = vy(n-1,b) + 0.5*dt*( ay(n,b) + ay(n-1,b) );
            vz(n,b) = vz(n-1,b) + 0.5*dt*( az(n,b) + az(n-1,b) );
        end
    end
end

%%

%Compute acceleration of all objects via Newtonian gravity
% Note x,y,z are in units of AU, M are in Earth masses
function [ax,ay,az] = newton( x,y,z,g,M,R,k )

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_earth = 5.97e24;
Yr = 365*24*3600;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G*Yr^2*M_earth/(AU^3);

%Number of bodies
B = length(M);

%Initialise accelerations
ax = zeros(1,B);
ay = zeros(1,B);
az = zeros(1,B);
for b=1:B
    for bb = 1:B
        %Ignore self reference to body b!
        if bb~=b
            
            %Compute inter-mass distance
            d = ( ( x(bb) - x(b) )^2 + ( y(bb) - y(b) )^2 + ( z(bb) - z(b))^2 )^(1/2);
            
            %Newton's Law of gravitation, unless r_cubed is less than sum
            %of object radii. In this case we will treat the objects as
            %elastic bodies colliding.
            if d > k*( R(b) + R(bb) )
                ax(b) = ax(b) + P*g * M(bb) * ( x(bb) - x(b) ) / d^3;
                ay(b) = ay(b) + P*g * M(bb) * ( y(bb) - y(b) ) / d^3;
                az(b) = az(b) + P*g * M(bb) * ( z(bb) - z(b) ) / d^3;
            else
                ax(b) = 0;
                ay(b) = 0;
                az(b) = 0;
            end
        end
    end
end

%End of code