function gravity_sim

%% INPUTS %%

vmax = 10;
AUmax = 8;
view_option = 3;

%Marker size for planets
msize = 2;

%Astronomical parameters in SI units
AU = 149.6e9;
G = 6.67e-11;
M_sun = 2e30;
Yr = 365*24*3600;

%Timestep / years
dt = 0.01;

%Strength of gravity
G_factor = 1;

%Dimensionless number which controls the dynamics, and results from the
%scaling of mass, distance and time parameters to make them dimensionless.
P = G_factor*G*Yr^2*M_sun/(AU^3);

%Compute initial orbital parameters for the two stars such that they rotate
%in a circular fashion about their mutual centre of mass

%Star masses in Solar masses
M1 = 1;
M2 = M1;
M3 = 0.1*M1;

%Star separation /AU
s = 5;

%Compute two star orbital parameters
[vx1,vx2,vy1,vy2,vz1,vz2,x1,y1,z1,x2,y2,z2,xx,yy,zz] =...
    two_body_circular_init( M1, M2, s, P );

%Star 1
masses(1).name = 'Sun';
masses(1).mass = M1;
masses(1).radii = 0.1;
masses(1).x0 = x1;
masses(1).y0 = y1;
masses(1).z0 = z1;
masses(1).vx0 = vx1;
masses(1).vy0 = vz1;
masses(1).vz0 = vy1;
masses(1).marker_RGB = [];

%Star 2
masses(2).name = 'Planet X';
masses(2).mass = M2;
masses(2).radii = 0.1;
masses(2).x0 = x2;
masses(2).y0 = y2;
masses(2).z0 = z2;
masses(2).vx0 = vx2;
masses(2).vy0 = vz2;
masses(2).vz0 = vy2;
masses(2).marker_RGB = [];

%Star 3
masses(3).name = 'Planet Y';
masses(3).mass = M3;
masses(3).radii = 0.1;
masses(3).x0 = 10;
masses(3).y0 = 0;
masses(3).z0 = 0;
masses(3).vx0 = -2;
masses(3).vy0 = -0.1;
masses(3).vz0 = 0;
masses(3).marker_RGB = [0,0,0];

%Rings and clusters of massless objects initially rotating about the two stars

%Ring 1
rings(1).xc = x1;
rings(1).yc = y1;
rings(1).zc = z1;
rings(1).vxc = vx1;
rings(1).vyc = vy1;
rings(1).vzc = vz1;
rings(1).num_rings = 15;
rings(1).arc_separation_AU = pi/30;
rings(1).first_ring_radius_AU = 1;
rings(1).ring_radius_diff_AU = 0.2;
rings(1).mass_at_centre = M1;
rings(1).marker_RGB = [1,0,0];

%Ring 2
rings(2).xc = x2;
rings(2).yc = y2;
rings(2).zc = z2;
rings(2).vxc = vx2;
rings(2).vyc = vy2;
rings(2).vzc = vz2;
rings(2).num_rings = 15;
rings(2).arc_separation_AU = pi/30;
rings(2).first_ring_radius_AU = 1;
rings(2).ring_radius_diff_AU = 0.2;
rings(2).mass_at_centre = M2;
rings(2).marker_RGB = [0,0,1];

%Clusters
clusters(1).xc = masses(2).x0;
clusters(1).yc = masses(2).y0;
clusters(1).zc = masses(2).z0;
clusters(1).vxc = masses(2).vx0;
clusters(1).vyc = masses(2).vy0;
clusters(1).vzc = masses(2).vz0;
clusters(1).shell_separation_AU = 0.2;      
clusters(1).num_shells = 10;                
clusters(1).num_masses_per_square_AU = 5; 
clusters(1).first_shell_radius = 1;         
clusters(1).mass_at_centre = M2;
clusters(1).marker_RGB = [1,0,1];

%Gravity sim title
title_str = ['M1 = ',num2str(M1),', M2 = ',num2str(M2),', M3 = ',num2str(M3) ];

%Run simulation
gsim( masses, rings, clusters, P, view_option, AUmax, vmax, title_str, dt, msize )

%%

% gsim
%
% Top level function which runs the gravity simulation. Inputs are defined
% in a script or function which refers to this function.
%
% LAST UPDATED by Andy French May 2012.
%
% Syntax: gravity_sim( masses, rings, clusters, view_option, title_str )
%
% masses(n).name              = Object string e.g. 'Earth'
% masses(n).mass              = Object mass in Solar masses
% masses(n).radii             = Object radii in AU
% masses(n).x0                = Object initial x coordinate in AU
% masses(n).y0                = Object initial y coordinate in AU
% masses(n).vz0               = Object initial z coordinate in AU
% masses(n).vx0               = Object initial x velocity coordinate in AU
% masses(n).vy0               = Object initial y velocity coordinate in AU
% masses(n).vz0               = Object initial z velocity coordinate in AU
% masses(n).marker_RGB        = Fix object colour. If [] then colour by speed
%
% rings(n).xc                    = x coordinate in AU of ring centre
% rings(n).yc                    = y coordinate in AU of ring centre
% rings(n).zc                    = z coordinate in AU of ring centre
% rings(n).vxc                   = Initial ring x velocity in AU per year
% rings(n).vyc                   = Initial ring y velocity in AU per year
% rings(n).vzc                   = Initial ring z velocity in AU per year
% rings(n).num_rings             = Number of rings of massless planets
% rings(n).arc_separation_AU     = Arc separation of planet in AU
% rings(n).first_ring_radius_AU  = Radius of the first ring in AU
% rings(n).ring_radius_diff_AU   = Difference in AU between rings
% rings(n).mass_at_centre        = Mass at centre of ring in Solar masses
% masses(n).marker_RGB           = Fix object colour. If [] then colour by speed
%
% clusters(n).xc                 = x coordinate in AU of cluster centre
% clusters(n).yc                 = y coordinate in AU of cluster centre
% clusters(n).zc                 = z coordinate in AU of cluster centre
% clusters(n).vxc                = Initial cluster x velocity in AU per year
% clusters(n).vyc                = Initial cluster y velocity in AU per year
% clusters(n).vzc                = Initial cluster z velocity in AU per year
% clusters(n).shell_separation_AU       = Separation of mass cluster spherical shells /AU
% clusters(n).num_shells                = Number of spherical shells
% clusters(n).num_masses_per_square_AU  = Number of planents per square AU in each shell
% clusters(n).first_shell_radius        = Radius of first shell in AU
% clusters(n).mass_at_centre        = Mass at centre of cluster in Solar masses
% clusters(n).marker_RGB            = Fix object colour. If [] then colour by speed
%
% P                          Dimensionless number which controls the dynamics, and results from the
%                            scaling of mass, distance and time parameters to make them dimensionless.
%                            P = G_factor*G*Yr^2*M_sun/(AU^3);
%
% view_option                View option. 2 means 2D view, 3 means 3D
%                            view, [az,el] means a prescribed azimuth and elevation in degrees.
%                            Essesntially the inputs to the view command.
%
% AUmax                      +/- plot limits in AU
% vmax                       Velocity limit in AU per year for colorbar.
%                            Note masses are coloured by their speed.
% title_str                  Title string
% dt                         Timestep /years
% msize                      Marker size of planets 

function gsim( masses, rings, clusters, P, view_option, AUmax, vmax, title_str, dt, msize )

% Assemble arrays for all masses and massless objects
x0 = [];
y0 = [];
z0 = [];
vx0 = [];
vy0 = [];
vz0 = [];
R = [];
M = [];

% Masses
if ~isempty(masses)
    for n=1:length(masses)
        x0 = [x0,masses(n).x0];
        y0 = [y0,masses(n).y0];
        z0 = [z0,masses(n).z0];
        vx0 = [vx0,masses(n).vx0];
        vy0 = [vy0,masses(n).vy0];
        vz0 = [vz0,masses(n).vz0];
        R = [R,masses(n).radii];
        M = [M,masses(n).mass];
    end
end

%Rings of massless planets
if ~isempty(rings)
    for n=1:length(rings)
        [xx,yy,zz,vxx,vyy,vzz,num_masses] = mass_rings( rings(n).xc,rings(n).yc,rings(n).zc,....
            rings(n).mass_at_centre, rings(n).arc_separation_AU,...
            rings(n).num_rings, rings(n).first_ring_radius_AU, rings(n).ring_radius_diff_AU, P);
        x0 = [x0,xx];
        y0 = [y0,yy];
        z0 = [z0,zz];
        vx0 = [vx0,vxx + rings(n).vxc];
        vy0 = [vy0,vyy + rings(n).vyc];
        vz0 = [vz0,vzz + rings(n).vzc];
        R = [R,zeros(1,num_masses)];
        M = [M,zeros(1,num_masses)];
        rings(n).num_masses = num_masses;
    end
end

%Globular clusters of massless planets
if ~isempty(clusters)
    for n=1:length(clusters)
        [xx,yy,zz,vxx,vyy,vzz,num_masses] = globular_cluster( clusters(n).xc,clusters(n).yc,clusters(n).zc,...
            clusters(n).mass_at_centre,clusters(n).shell_separation_AU,...
            clusters(n).num_shells, clusters(n).first_shell_radius, clusters(n).num_masses_per_square_AU, P);
        x0 = [x0,xx];
        y0 = [y0,yy];
        z0 = [z0,zz];
        vx0 = [vx0,vxx + clusters(n).vxc];
        vy0 = [vy0,vyy + clusters(n).vyc];
        vz0 = [vz0,vzz + clusters(n).vzc];
        R = [R,zeros(1,num_masses)];
        M = [M,zeros(1,num_masses)];
        clusters(n).num_masses = num_masses;
    end
end

%Coefficients of restitutia (MODIFY THIS, and bounce condition - TOO COMPLEX FOR THIS SIMULATION)
C = ones(length(R),length(R));

%%

%% GRAVITY SIM ANIMATION %%

%Set up figure
fig = figure('color',[1 1 1],'name','gravity simulation animation',...
    'renderer','painters','KeyPressFcn',@keypressfunc,'units','normalized','position',[0 0 1 1]);
d = get(fig,'userdata');
d.run = 1;
d.save = 0;
d.print = 0;
d.wait = 0;
t = 0;
set(fig,'userdata',d);

%Set up axis
cmap = speed_colormap;
colormap(cmap);
colorbar
caxis([0,vmax])
hold on;
xlabel('x /AU')
ylabel('y /AU')
zlabel('z /AU')
tit = title({['Gravity sim: ',title_str],['t = ',num2str(t),' years.']});
view(view_option)
axis vis3d
axis equal
xlim([-AUmax,AUmax])
ylim([-AUmax,AUmax])
zlim([-AUmax,AUmax])
grid on;
hold on;

%Initialise object data vectors
x = x0;
y = y0;
z = z0;
vx = vx0;
vy = vy0;
vz = vz0;

%Plot initial positions. Colour code by velocity

%Masses
if ~isempty(masses)
    for n=1:length(masses)
        if isempty( masses(n).marker_RGB )
            [r,g,b] = x_to_color(sqrt(vx(n).^2 + vy(n).^2 + vz(n).^2 ),cmap,0,vmax);
        else
            r = masses(n).marker_RGB(1);
            g = masses(n).marker_RGB(2);
            b = masses(n).marker_RGB(3);
        end
        p(n) = plot3(x0(n),y0(n),z0(n),'ro','markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Rings
if ~isempty(rings)
    n0 = length(masses);
    end_index = n0;
    for n=1:length(rings)
        start_index = end_index+1;
        end_index = start_index + rings(n).num_masses - 1;
        if isempty( rings(n).marker_RGB )
            [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
        else
            r = rings(n).marker_RGB(1);
            g = rings(n).marker_RGB(2);
            b = rings(n).marker_RGB(3);
        end
        p(n0+n) = plot3(x0(start_index:end_index),y0(start_index:end_index),z0(start_index:end_index),'r.','markersize',msize,...
            'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Clusters
if ~isempty(clusters)
    n0 = n0 + length(rings);
    for n=1:length(clusters)
        start_index = end_index+1;
        end_index = start_index + clusters(n).num_masses - 1;
        if isempty( clusters(n).marker_RGB )
            [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
        else
            r = clusters(n).marker_RGB(1);
            g = clusters(n).marker_RGB(2);
            b = clusters(n).marker_RGB(3);
        end
        p(n0+n) = plot3(x0(start_index:end_index),y0(start_index:end_index),z0(start_index:end_index),'r.','markersize',msize,...
            'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
    end
end

%Run simulation until user presses 'q'
while d.run == 1
    %Obtain figure user data and check for key presses
    d = get(fig,'userdata');
    if d.wait == 1
        pause(0.1);
    elseif d.wait == 0
        
        %Update positions, veocities and accelerations and times
        [ x, y, z, vx, vy, vz, ax, ay, az ] =...
            gravity( x, y, z, vx, vy, vz, R, M, P, dt, 2 );
        t = t + dt;
        x = x(2,:);
        y = y(2,:);
        z = z(2,:);
        vx = vx(2,:);
        vy = vy(2,:);
        vz = vz(2,:);
        
        %Masses
        if ~isempty(masses)
            for n=1:length(masses)
                if isempty( masses(n).marker_RGB )
                    [r,g,b] = x_to_color(sqrt(vx(n).^2 + vy(n).^2 + vz(n).^2 ),cmap,0,vmax);
                else
                    r = masses(n).marker_RGB(1);
                    g = masses(n).marker_RGB(2);
                    b = masses(n).marker_RGB(3);
                end
                set( p(n), 'Xdata', x(n), 'Ydata', y(n), 'Zdata', z(n),'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
                plot3(x(n),y(n),z(n),'r.','markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b],'markersize',1);
            end
        end
        
        %Rings
        if ~isempty(rings)
            n0 = length(masses);
            end_index = n0;
            for n=1:length(rings)
                start_index = end_index+1;
                end_index = start_index + rings(n).num_masses - 1;
                if isempty( rings(n).marker_RGB )
                    [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                        vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
                else
                    r = rings(n).marker_RGB(1);
                    g = rings(n).marker_RGB(2);
                    b = rings(n).marker_RGB(3);
                end
                set( p(n0+n), 'Xdata', x(start_index:end_index), 'Ydata', y(start_index:end_index), 'Zdata',...
                    z(start_index:end_index),...
                    'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
            end
        end
        
        %Clusters
        if ~isempty(clusters)
            n0 = n0 + length(rings);
            for n=1:length(clusters)
                start_index = end_index+1;
                end_index = start_index + clusters(n).num_masses - 1;
                if isempty( clusters(n).marker_RGB )
                    [r,g,b] = x_to_color( mean( sqrt(vx(start_index:end_index).^2 +...
                        vy(start_index:end_index).^2 + vz(start_index:end_index).^2 )),cmap,0,vmax);
                else
                    r = clusters(n).marker_RGB(1);
                    g = clusters(n).marker_RGB(2);
                    b = clusters(n).marker_RGB(3);
                end
                set( p(n0+n), 'Xdata', x(start_index:end_index), 'Ydata', y(start_index:end_index), 'Zdata',...
                    z(start_index:end_index),...
                    'markerfacecolor',[r,g,b],'markeredgecolor',[r,g,b]);
            end
        end
        
        %Update title
        set(tit,'string',{['Gravity sim: ',title_str],['t = ',num2str(t),' years.']})
        
        %Flush pending graphics requests
        drawnow
        
        %Special actions resulting from key presses
        d = get(fig,'userdata');
        if d.save == 1
            %Save a .mat file of the current data set
            save( ['gravity_sim_data t=',num2str(t),'.mat'] );
            d.save = 0;
        end
        if d.print == 1;
            %Print a screenshot of the current view
            print( fig, '-dpng','-r300',['gravity sim. t=',num2str(t),...
                '.  ',strrep(datestr(now),':','-'),'.png'])
            d.print = 0;
        end
        set(fig,'userdata',d);
    end
end
close(fig);

%%

%Figure key press function callback
function keypressfunc( fig,evnt )
d = get(fig,'userdata');
if evnt.Character == 'q'
    d.run = 0;
elseif evnt.Character == 'p'
    d.print = 1;
elseif evnt.Character == 's'
    d.save = 1;
elseif evnt.Character == 'w'
    d.wait = 1;
elseif evnt.Character == 'c'
    d.wait = 0;
end
set(fig,'userdata',d);

%%

%mass_rings
% Function which creates a vectors of x,y,z coordinates dscribing the
% initial position of rings of masses orbiting in a circular fashion about
% a centra mass mc. vx,vy and vz are the corresponding x,y,z velocities.

function [x0,y0,z0,vx0,vy0,vz0,num_masses] = mass_rings( xc,yc,zc,mc,arc_separation_AU,...
    num_rings, r0, ring_radius_diff_AU, P)

%Planets starting their orbits in concentric rings about star with coordinates
%(xc,yc,zc)
x0 = [];
y0 = [];
z0 = [];
vx0 = [];
vy0 = [];
vz0 = [];
for n=1:num_rings
    
    %Ring radius /AU
    r = r0 + ring_radius_diff_AU*(n-1);
    
    %Ring prbital speed /AU per Year
    v = sqrt( P *mc/r );
    
    %Number of planets per ring
    num_planets_per_ring = floor( 2*pi*r/arc_separation_AU );
    
    %Compute planet positions and velocities
    if num_planets_per_ring>=1
        theta = linspace(0,2*pi,num_planets_per_ring+1);
        for k=1:num_planets_per_ring
            x0 = [x0,xc+r*cos(theta(k))];
            vx0 = [vx0,-v*sin(theta(k))];
            y0 = [y0,yc+r*sin(theta(k))];
            vy0 = [vy0,v*cos(theta(k))];
            z0 = [z0,zc];
            vz0 = [vz0,0];
        end
    end
end

%Compute total number of masses
num_masses = length(x0);

%%

%gravity
% N-body gravity simulator. Uses the Verlet algorithm + Newtonian gravity +
% repulsion to determine future x,y,z coordinates (and velocities and accelerations)
% of N masses.
%
% [ t, x, y, z, vx, vy, vz, ax, ay, az ] =...
%    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, P, dt, N )
%
% t               Vector of simulation times / Earth years
% x,y,z           x,y,z coordinates in AU (distance between the Earth and
%                 the Sun
% vx,vy,vz        Cartesian velocities in AU per Earth year
% ax,ay,az        Cartesian accelerations in AU per square Earth years.
%
% x0,y0,z0        Initial object positions in AU
% vx0,vy0,vz0     Initial ibject velocities in AU per Earth year
% R               Row vector of object raddi / Sun radii
% M               Row vector of object masses / Sun masses
% P               Gravity constant
% dt              Time step, in  Earth years
% N               Number of time steps

function [ x, y, z, vx, vy, vz, ax, ay, az ] =...
    gravity( x0, y0, z0, vx0, vy0, vz0, R, M, P, dt, N )

%Number of bodies
B = length(x0);

%Initialise output arrays
x = repmat( x0, [N,1] );
y = repmat( y0, [N,1] );
z = repmat( z0, [N,1] );
vx = repmat( vx0, [N,1] );
vy = repmat( vy0, [N,1] );
vz = repmat( vz0, [N,1] );

%Work out inital acceleration
[ax0,ay0,az0] = newton( x0,y0,z0,P,M,R );
ax = repmat( ax0, [N,1] );
ay = repmat( ay0, [N,1] );
az = repmat( az0, [N,1] );

%Work out subsequent dynamics of all bodies via Verlet + Newtonian Gravity
for n=2:N
    
    %Update positions
    x(n,:) = x(n-1,:) + dt * vx(n-1,:) + 0.5*dt*dt*ax(n-1,:);
    y(n,:) = y(n-1,:) + dt * vy(n-1,:) + 0.5*dt*dt*ay(n-1,:);
    z(n,:) = z(n-1,:) + dt * vz(n-1,:) + 0.5*dt*dt*az(n-1,:);
    
    %Update acceleration
    [Ax,Ay,Az] = newton( x(n,:),y(n,:),z(n,:),P,M,R );
    ax(n,:) = Ax(:);
    ay(n,:) = Ay(:);
    az(n,:) = Az(:);
    
    %Update velocity of objects using acceleration
    vx(n,:) = vx(n-1,:) + 0.5*dt*( ax(n,:) + ax(n-1,:) );
    vy(n,:) = vy(n-1,:) + 0.5*dt*( ay(n,:) + ay(n-1,:) );
    vz(n,:) = vz(n-1,:) + 0.5*dt*( az(n,:) + az(n-1,:) );
end

%%

%Compute acceleration of all objects via Newtonian-ish gravity
% Note x,y,z are in units of AU, M are in Earth masses
function [ax,ay,az] = newton( x,y,z,P,M,R )

%Gravitation exponent
Ng = 2;

%Repulsion exponent
Nr = 5;

%Number of bodies
B = length(M);

%Work out bodies which don't have zero mass
nzm = find(M~=0);
zm = find(M==0);

%Initialise accelerations
ax = zeros(1,B);
ay = zeros(1,B);
az = zeros(1,B);

%

%Compute acceleration of non-zero masses

%Compute accelerations resulting from each mass and store these in a matrix aa.
d = distance( [x(nzm);y(nzm);z(nzm)],[x(nzm);y(nzm);z(nzm)] );
m = repmat( M(nzm).',[1,length(nzm)] );

%Compute repulsion constant
rc = repmat( R(nzm).',[1,length(nzm)] );

%x component of acceleration
aa = m.*( repmat( x(nzm).',[1,length(nzm)] ) - repmat( x(nzm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( x(nzm).',[1,length(nzm)] ) - repmat( x(nzm),[length(nzm),1] ))./d.^(Nr+1);
aa(1:length(nzm)+1:end) = 0;  %Don't let self same mass displacements contribute!
ax(nzm) = P*sum(aa,1);

%y component of acceleration
aa = m.*( repmat( y(nzm).',[1,length(nzm)] ) - repmat( y(nzm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( y(nzm).',[1,length(nzm)] ) - repmat( y(nzm),[length(nzm),1] ))./d.^(Nr+1);
aa(1:length(nzm)+1:end) = 0; %Don't let self same mass displacements contribute!
ay(nzm) = P*sum(aa,1);

%z component of acceleration
aa = m.*( repmat( z(nzm).',[1,length(nzm)] ) - repmat( z(nzm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( z(nzm).',[1,length(nzm)] ) - repmat( z(nzm),[length(nzm),1] ))./d.^(Nr+1);
aa(1:length(nzm)+1:end) = 0; %Don't let self same mass displacements contribute!
az(nzm) = P*sum(aa,1);

%

%Compute acceleration of zero masses

%Compute accelerations resulting from each mass and store these in a matrix aa.
d = distance( [x(nzm);y(nzm);z(nzm)],[x(zm);y(zm);z(zm)] );
m = repmat( M(nzm).',[1,length(zm)] );

%Compute repulsion constant
rc = repmat( R(nzm).',[1,length(zm)] );

%x component of acceleration
aa = m.*( repmat( x(nzm).',[1,length(zm)] ) - repmat( x(zm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( x(nzm).',[1,length(zm)] ) - repmat( x(zm),[length(nzm),1] ))./d.^(Nr+1);
ax(zm) = P*sum(aa,1);

%y component of acceleration
aa = m.*( repmat( y(nzm).',[1,length(zm)] ) - repmat( y(zm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( y(nzm).',[1,length(zm)] ) - repmat( y(zm),[length(nzm),1] ))./d.^(Nr+1);
ay(zm) = P*sum(aa,1);

%z component of acceleration
aa = m.*( repmat( z(nzm).',[1,length(zm)] ) - repmat( z(zm),[length(nzm),1] ))./d.^(Ng+1) -...
    rc.*m.*( repmat( z(nzm).',[1,length(zm)] ) - repmat( z(zm),[length(nzm),1] ))./d.^(Nr+1);
az(zm) = P*sum(aa,1);

%%

%two_body_circular_init
% Function which returns initial conditions for mutually circular orbits
% given masses m1,m2 (in Earth masses) and separation s in AU.
% Initial x,y,z coordinates are in AU and velocities are in AU per Year.
%
% The centre of mass of the system is defined to be the origin of the
% coordinate systems.
%
% LAST UPDATED by Andy French May 2012.

function [vx1,vx2,vy1,vy2,vz1,vz2,x1,y1,z1,x2,y2,z2,xx,yy,zz] =...
    two_body_circular_init( m1, m2, s, P )

%Define initial position coordinates
x1 = 0;
x2 = s;
y1 = 0;
y2 = 0;
z1 = 0;
z2 = 0;

%Compute centre of mass coordinates
xx = ( m1*x1 + m2*x2 )/(m1+m2);
yy = ( m1*y1 + m2*y2 )/(m1+m2);
zz = ( m1*z1 + m2*z2 )/(m1+m2);

%Shift coordinate system so centre of mass is the origin
x1 = x1 - xx;
x2 = x2 - xx;
y1 = y1 - yy;
y2 = y2 - yy;
z1 = z1 - zz;
z2 = z2 - zz;

%Define initial velocity coordinates to result in circular orbits in the
%x,y plane
vx1 = 0;
vx2 = 0;
vy1 = sqrt( P * (m1 + m2) * xx^2 / s^3 );
vy2 = -sqrt( P * (m1 + m2) * (s-xx)^2 / s^3 );
vz1 = 0;
vz2 = 0;

%distance
% Function which computes Euclidean distance matrix.
% This fully vectorized (VERY FAST!) m-file computes the
% Euclidean distance between two vectors by:
%          ||A-B|| = sqrt ( ||A||^2 + ||B||^2 - 2*A.B )
%
% Syntax: E = distance(A,B)
%    A - (DxM) matrix
%    B - (DxN) matrix
%    E - (MxN) Euclidean distances between vectors in A and B
%
% Example :
%    A = rand(400,100); B = rand(400,200);
%    d = distance(A,B);

% Author   : Roland Bunschoten
%            University of Amsterdam
%            Intelligent Autonomous Systems (IAS) group
%            Kruislaan 403  1098 SJ Amsterdam
%            tel.(+31)20-5257524
%            bunschot@wins.uva.nl
% Last Rev : Oct 29 16:35:48 MET DST 1999
% Tested   : PC Matlab v5.2 and Solaris Matlab v5.3
% Thanx    : Nikos Vlassis

function d = distance(a,b)
if (nargin ~= 2)
    error('Not enough input arguments');
end
if (size(a,1) ~= size(b,1))
    error('A and B should be of same dimensionality');
end
aa=sum(a.*a,1); bb=sum(b.*b,1); ab=a'*b;
d = sqrt(abs(repmat(aa',[1 size(bb,2)]) + repmat(bb,[size(aa,2) 1]) - 2*ab));

%%

%interp_colormap
% Function which interpolates current colourmap to yield better graduated
% shading. N is number of possible colours.
%
% LAST UPDATED by Andy French 14 September 2008

function interp_colormap( N )

%Get current colourmap
map = colormap;

%Initialise new colormap
new_map = ones(N,3);

%Get size of current colormap and initalise red,green,blue vectors
dim = size(map);
R = ones(1,dim(1));
G = ones(1,dim(1));
B = ones(1,dim(1));
RR = ones(1,N);
GG = ones(1,N);
BB = ones(1,N);

%Populate these with current colormap
R(:) = map(:,1);
G(:) = map(:,2);
B(:) = map(:,3);

%Interpolate to yield new colour map
x = linspace( 1, dim(1), N );
RR = interp1( 1:dim(1), R, x );
GG = interp1( 1:dim(1), G, x );
BB = interp1( 1:dim(1), B, x );
new_map(:,1) = RR(:);
new_map(:,2) = GG(:);
new_map(:,3) = BB(:);

%Set colormap to be new map
colormap( new_map );

%%

%x_to_color
%
% LAST UPDATED by Andy French May 2012

function [r,g,b] = x_to_color(x,cmap,xmin,xmax)

red = cmap(:,1).';
green = cmap(:,2).';
blue = cmap(:,3).';
cx = linspace( 0, 1, length(red) );
x = (x - xmin)/(xmax - xmin);
x(x>1) = 0.99;
x(x<0) = 0.01;
r = interp1( cx, red, x );
g = interp1( cx, green, x );
b = interp1( cx, blue, x );

%%

%speed_colormap

function map = speed_colormap

N = 200;
x = linspace(0,1,N);

%Start color
R1 = 0;
G1 = 0;
B1 = 1;

%End color
R2 = 1;
G2 = 0.4;
B2 = 0;

%Create interpolated colormap
R = interp1([0,1],[R1,R2],x).';
G = interp1([0,1],[G1,G2],x).';
B = interp1([0,1],[B1,B2],x).';
map = [R,G,B];

%%

%globular_cluster
% Function which creates arrays of positions and velocities of masses that
% will form circular orbits aboyt a central mass, fillig a spherical volume
% around the central mass.
%
% LAST UPDATED by Andy French May 2012

function [x0,y0,z0,vx0,vy0,vz0,num_masses] = globular_cluster( xc,yc,zc,mc,shell_separation_AU,...
    num_shells, r0, num_masses_per_square_AU, P)

%Planets starting their orbits in spherical shells about star with coordinates
%(xc,yc,zc)
x0 = [];
y0 = [];
z0 = [];
vx0 = [];
vy0 = [];
vz0 = [];
for n=1:num_shells
    
    %Ring radius /AU
    r = r0 + shell_separation_AU*(n-1);
    
    %Ring prbital speed /AU per Year
    v = sqrt( P *mc/r );
    
    %Number of planets per shell
    s = floor( sqrt( num_masses_per_square_AU*4*pi*r^2 ));
    azi = linspace(0,2*pi,s+1);
    elev = linspace(-pi,pi,s+1);
    
    %Compute planet positions and velocities
    if s >=1
        for i=1:s
            for j=1:s
                %x
                x0 = [x0,xc+r*cos(elev(i))*sin(azi(j))];
                vx0 = [vx0,-v*sin(elev(i))*sin(azi(j))];
                
                %y
                y0 = [y0,yc+r*cos(elev(i))*cos(azi(j))];
                vy0 = [vy0,-v*sin(elev(i))*cos(azi(j))];
                
                %z
                z0 = [z0,zc + r*sin(elev(i))];
                vz0 = [vz0,v*cos(elev(i))];
            end
        end
    end
end

%Compute total number of masses
num_masses = length(x0);

%End of code